﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class DetachVolumeRequest : public EC2Request {
 public:
  AWS_EC2_API DetachVolumeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DetachVolume"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The device name.</p>
   */
  inline const Aws::String& GetDevice() const { return m_device; }
  inline bool DeviceHasBeenSet() const { return m_deviceHasBeenSet; }
  template <typename DeviceT = Aws::String>
  void SetDevice(DeviceT&& value) {
    m_deviceHasBeenSet = true;
    m_device = std::forward<DeviceT>(value);
  }
  template <typename DeviceT = Aws::String>
  DetachVolumeRequest& WithDevice(DeviceT&& value) {
    SetDevice(std::forward<DeviceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Forces detachment if the previous detachment attempt did not occur cleanly
   * (for example, logging into an instance, unmounting the volume, and detaching
   * normally). This option can lead to data loss or a corrupted file system. Use
   * this option only as a last resort to detach a volume from a failed instance. The
   * instance won't have an opportunity to flush file system caches or file system
   * metadata. If you use this option, you must perform file system check and repair
   * procedures.</p>
   */
  inline bool GetForce() const { return m_force; }
  inline bool ForceHasBeenSet() const { return m_forceHasBeenSet; }
  inline void SetForce(bool value) {
    m_forceHasBeenSet = true;
    m_force = value;
  }
  inline DetachVolumeRequest& WithForce(bool value) {
    SetForce(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the instance. If you are detaching a Multi-Attach enabled volume,
   * you must specify an instance ID.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  DetachVolumeRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the volume.</p>
   */
  inline const Aws::String& GetVolumeId() const { return m_volumeId; }
  inline bool VolumeIdHasBeenSet() const { return m_volumeIdHasBeenSet; }
  template <typename VolumeIdT = Aws::String>
  void SetVolumeId(VolumeIdT&& value) {
    m_volumeIdHasBeenSet = true;
    m_volumeId = std::forward<VolumeIdT>(value);
  }
  template <typename VolumeIdT = Aws::String>
  DetachVolumeRequest& WithVolumeId(VolumeIdT&& value) {
    SetVolumeId(std::forward<VolumeIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline DetachVolumeRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_device;

  bool m_force{false};

  Aws::String m_instanceId;

  Aws::String m_volumeId;

  bool m_dryRun{false};
  bool m_deviceHasBeenSet = false;
  bool m_forceHasBeenSet = false;
  bool m_instanceIdHasBeenSet = false;
  bool m_volumeIdHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
