﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/securityhub/SecurityHubRequest.h>
#include <aws/securityhub/SecurityHub_EXPORTS.h>
#include <aws/securityhub/model/AutomationRulesAction.h>
#include <aws/securityhub/model/AutomationRulesFindingFilters.h>
#include <aws/securityhub/model/RuleStatus.h>

#include <utility>

namespace Aws {
namespace SecurityHub {
namespace Model {

/**
 */
class CreateAutomationRuleRequest : public SecurityHubRequest {
 public:
  AWS_SECURITYHUB_API CreateAutomationRuleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateAutomationRule"; }

  AWS_SECURITYHUB_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> User-defined tags associated with an automation rule. </p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateAutomationRuleRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateAutomationRuleRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Whether the rule is active after it is created. If this parameter is equal
   * to <code>ENABLED</code>, Security Hub starts applying the rule to findings and
   * finding updates after the rule is created. To change the value of this parameter
   * after creating a rule, use <a
   * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateAutomationRules.html">
   * <code>BatchUpdateAutomationRules</code> </a>. </p>
   */
  inline RuleStatus GetRuleStatus() const { return m_ruleStatus; }
  inline bool RuleStatusHasBeenSet() const { return m_ruleStatusHasBeenSet; }
  inline void SetRuleStatus(RuleStatus value) {
    m_ruleStatusHasBeenSet = true;
    m_ruleStatus = value;
  }
  inline CreateAutomationRuleRequest& WithRuleStatus(RuleStatus value) {
    SetRuleStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An integer ranging from 1 to 1000 that represents the order in which the rule
   * action is applied to findings. Security Hub applies rules with lower values for
   * this parameter first. </p>
   */
  inline int GetRuleOrder() const { return m_ruleOrder; }
  inline bool RuleOrderHasBeenSet() const { return m_ruleOrderHasBeenSet; }
  inline void SetRuleOrder(int value) {
    m_ruleOrderHasBeenSet = true;
    m_ruleOrder = value;
  }
  inline CreateAutomationRuleRequest& WithRuleOrder(int value) {
    SetRuleOrder(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The name of the rule. </p>
   */
  inline const Aws::String& GetRuleName() const { return m_ruleName; }
  inline bool RuleNameHasBeenSet() const { return m_ruleNameHasBeenSet; }
  template <typename RuleNameT = Aws::String>
  void SetRuleName(RuleNameT&& value) {
    m_ruleNameHasBeenSet = true;
    m_ruleName = std::forward<RuleNameT>(value);
  }
  template <typename RuleNameT = Aws::String>
  CreateAutomationRuleRequest& WithRuleName(RuleNameT&& value) {
    SetRuleName(std::forward<RuleNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A description of the rule. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateAutomationRuleRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether a rule is the last to be applied with respect to a finding
   * that matches the rule criteria. This is useful when a finding matches the
   * criteria for multiple rules, and each rule has different actions. If a rule is
   * terminal, Security Hub applies the rule action to a finding that matches the
   * rule criteria and doesn't evaluate other rules for the finding. By default, a
   * rule isn't terminal. </p>
   */
  inline bool GetIsTerminal() const { return m_isTerminal; }
  inline bool IsTerminalHasBeenSet() const { return m_isTerminalHasBeenSet; }
  inline void SetIsTerminal(bool value) {
    m_isTerminalHasBeenSet = true;
    m_isTerminal = value;
  }
  inline CreateAutomationRuleRequest& WithIsTerminal(bool value) {
    SetIsTerminal(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A set of ASFF finding field attributes and corresponding expected values
   * that Security Hub uses to filter findings. If a rule is enabled and a finding
   * matches the conditions specified in this parameter, Security Hub applies the
   * rule action to the finding. </p>
   */
  inline const AutomationRulesFindingFilters& GetCriteria() const { return m_criteria; }
  inline bool CriteriaHasBeenSet() const { return m_criteriaHasBeenSet; }
  template <typename CriteriaT = AutomationRulesFindingFilters>
  void SetCriteria(CriteriaT&& value) {
    m_criteriaHasBeenSet = true;
    m_criteria = std::forward<CriteriaT>(value);
  }
  template <typename CriteriaT = AutomationRulesFindingFilters>
  CreateAutomationRuleRequest& WithCriteria(CriteriaT&& value) {
    SetCriteria(std::forward<CriteriaT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> One or more actions to update finding fields if a finding matches the
   * conditions specified in <code>Criteria</code>. </p>
   */
  inline const Aws::Vector<AutomationRulesAction>& GetActions() const { return m_actions; }
  inline bool ActionsHasBeenSet() const { return m_actionsHasBeenSet; }
  template <typename ActionsT = Aws::Vector<AutomationRulesAction>>
  void SetActions(ActionsT&& value) {
    m_actionsHasBeenSet = true;
    m_actions = std::forward<ActionsT>(value);
  }
  template <typename ActionsT = Aws::Vector<AutomationRulesAction>>
  CreateAutomationRuleRequest& WithActions(ActionsT&& value) {
    SetActions(std::forward<ActionsT>(value));
    return *this;
  }
  template <typename ActionsT = AutomationRulesAction>
  CreateAutomationRuleRequest& AddActions(ActionsT&& value) {
    m_actionsHasBeenSet = true;
    m_actions.emplace_back(std::forward<ActionsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Map<Aws::String, Aws::String> m_tags;

  RuleStatus m_ruleStatus{RuleStatus::NOT_SET};

  int m_ruleOrder{0};

  Aws::String m_ruleName;

  Aws::String m_description;

  bool m_isTerminal{false};

  AutomationRulesFindingFilters m_criteria;

  Aws::Vector<AutomationRulesAction> m_actions;
  bool m_tagsHasBeenSet = false;
  bool m_ruleStatusHasBeenSet = false;
  bool m_ruleOrderHasBeenSet = false;
  bool m_ruleNameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_isTerminalHasBeenSet = false;
  bool m_criteriaHasBeenSet = false;
  bool m_actionsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SecurityHub
}  // namespace Aws
